/******************************************************************************
 * (C) Copyright 2000 by Agilent Technologies GmbH. All rights reserved.      *
 ******************************************************************************/


/* ---------------------------------------------------------------
 * File: xfhif32.c 
 *       Functions used to access the Fast-Host-Interface card
 * -----------------------------------------------------------------*/

/* --------------------------------------------------------------------------
 *
 * ABSTRACT: Functions used to access the Fast-Host-Interface card
 * These functions are mostly called by wrapper functions in iocommon.c
 * which switch between the different ports.
 *
 * NOTES:    Win95 or WinNT PCI-drivers ONLY
 * Although the FastHIF has its own PCI driver (bx_nthif) which we use for 
 * reads/writes we also use the gen. purpose BestX PCI driver (bx_ntpcix) to get 
 * slotdata and  device id's for different FastHIF cards.
 *
 * Therefore bx_ntpcix.sys MUST BE ACTIVE ON THE HOST SYSTEM !!!
 *
 * For HIF the bx_handlearray has the following meanings
 * bx_handlearray[].port            BX_PORT_FASTHIF
 * bx_handlearray[].portnumber      Win32 FileHandle (returned by CreateFile())
 * bx_handlearray[].entered_port    HPSlotId (from registry..put there by bx_ntpci.sys)
 * -------------------------------------------------------------------------- */


#include <xtypedef.h>
#include <stdio.h>
#include <windows.h>
#include <winioctl.h>

#include <b_hif.h>

#include <xhif.h>
#include <xcmd.h>
#include <xsession.h>
#include <xlasterr.h>
#include <xidentos.h>
#include <timeout.h>

#include <assert.h>

extern bx_handlestype bx_handlearray[];

/* --------------------------------------------------------------------------
 * STATICS
 * -------------------------------------------------------------------------- */

/* --------------------------------------------------------------------------
 * BestXHIFDeviceNameGet() gets the name that we'll use to open a HIF device.  
 * Unlike the regular PCI the name is not enumerated in the registry and 
 * varies between OS's.  NameBuffer should be 64 bytes long.
 * -------------------------------------------------------------------------- */

static bx_errtype BestXHIFDeviceNameGet(bx_int32 dwCardIndex, char * pNameBuffer)
{
  switch ( GetOsVersionShort() )
  {
    case OS_WINNT:
    case OS_WIN2K:
    case OS_WIN95:
      BESTX_SPRINTF(pNameBuffer, "\\\\.\\%s%d", HIF_DOS_DEVICE_NAME, dwCardIndex);
      break;
  
/*    case OS_WIN95:
       * NOTE; the win95 driver can only handle 1 card...thus we will
       * come up with an illegal name if dwCardIndex is anything but 0 *
      if(dwCardIndex)
        BESTX_SPRINTF(pNameBuffer, "\\\\.\\%s%d.VXD", HIF_WIN95_DEVICE_NAME, dwCardIndex);
      else
        BESTX_SPRINTF(pNameBuffer, "\\\\.\\%s.VXD", HIF_WIN95_DEVICE_NAME);

      break;
*/  
    default:
      return BX_E_HIF_WIN32_ONLY;
  }
  return BX_E_OK;
}



/* --------------------------------------------------------------------------
 * CAPI-only functions 
 * -------------------------------------------------------------------------- */

/* --------------------------------------------------------------------------
 * The register width of the Fast HIF is always 1
 * -------------------------------------------------------------------------- */

/* --------------------------------------------------------------------------
 * Open a FastHIF card device.  Note that CardId is identical to CardIndex !!
 * The OS Handle and HPSlotId are both returned for storing in the handle
 * array if the open is successful.
 * -------------------------------------------------------------------------- */

bx_errtype BestXOpenHIF(bx_int32 dwCardId, 
                      bx_int32 * pHPSlotId, 
                      bx_portnumtype * pOsHandle )
{

#define     MAX_CREATE_NAME_LEN     64

  HANDLE OsHandle;

  char cDeviceName[MAX_CREATE_NAME_LEN];
  bx_int32 dwBuffLen = MAX_CREATE_NAME_LEN;
  bx_errtype err;

  assert(pHPSlotId && pOsHandle && "Null ptr. passed to BestXOpenHIF()");

  /* init for failure */
  *pOsHandle = INVALID_OS_HANDLE;

  /* **TODO** check the version here */
  /*
  if (BX_E_OK != (err = BestXHIFVersionCheck()))
  {
    return err;
  }
   */

  /* get the Win32 name that we'll use to open the device. */
  if (BX_E_OK != (err = BestXHIFDeviceNameGet(dwCardId, cDeviceName)))
  {
    return err;
  }

  /* now try to open the device */
  OsHandle = CreateFile(cDeviceName,
                        GENERIC_WRITE | GENERIC_READ,
                        0,  /* was FILE_SHARE_WRITE */
                        NULL,
                        OPEN_EXISTING,
                        0,
                        NULL);

  if (INVALID_HANDLE_VALUE == OsHandle)
  {
    DBG_ApiLastError;
    return BX_E_HIF_OPEN;
  }

  *pOsHandle = (bx_portnumtype) OsHandle;

  return BX_E_OK;
}


/* --------------------------------------------------------------------------
 * 
 * -------------------------------------------------------------------------- */

bx_errtype BestXCloseHIF(bx_portnumtype OsHandle)
{
  /* OS call to close device */
  if (!CloseHandle((HANDLE) OsHandle))
  {
    return BX_E_BAD_HANDLE;
  }

  return BX_E_OK;
}

/* --------------------------------------------------------------------------
 * 
 * -------------------------------------------------------------------------- */

bx_errtype BestXHIFDeviceConnect(bx_portnumtype OsHandle)
{
  BOOL fIoCtlSuccess;
  DWORD   dwReturned=0;

  fIoCtlSuccess = DeviceIoControl(
                    (HANDLE) OsHandle,          /* Handle to device  */
                    (ULONG) IOCTL_HIF_CONNECT,  /* IO Control code for F-HIF card */
                    NULL,                       /* Buffer to driver. */
                    0,                          /* Length of buffer in bytes. */
                    NULL,                       /* Buffer from driver.  */
                    0,                          /* Length of buffer in bytes. */
                    &dwReturned,                /* Bytes placed in DataBuffer. */
                    NULL                        /* NULL means wait till op. completes. */
                    );

  DBG_ApiFailureIsZero(fIoCtlSuccess);

#if 1
  /* Driver always returns no success, ignore return value. It will be
     caught by connection check anyway. CZ */
  if (!fIoCtlSuccess)
    return BX_E_NOT_CONNECTED;
#endif
  
  return BX_E_OK;
}


/* --------------------------------------------------------------------------
 * 
 * -------------------------------------------------------------------------- */

void BestXHIFReleaseConnection(bx_portnumtype OsHandle)
{
  BOOL fIoCtlSuccess;
  ULONG ReturnedLength = 0;

  fIoCtlSuccess = DeviceIoControl(
                    (HANDLE) OsHandle,          /* Handle to device  */
                    (ULONG) IOCTL_HIF_DISCONNECT,  /* IO Control code for F-HIF card */
                    NULL,                       /* Buffer to driver. */
                    0,                          /* Length of buffer in bytes. */
                    NULL,                       /* Buffer from driver.  */
                    0,                          /* Length of buffer in bytes. */
                    &ReturnedLength,            /* Bytes placed in DataBuffer. */
                    NULL                        /* NULL means wait till op. completes. */
                    );

  DBG_ApiFailureIsZero(fIoCtlSuccess);

  return;
}


/* --------------------------------------------------------------------------
 * Note; All this does is talk to the HIF (not to the BestX)
 * -------------------------------------------------------------------------- */

bx_errtype BestXHIFCheckConnection(bx_portnumtype OsHandle)
{
  BOOL fIoCtlSuccess;
  DWORD   dwReturned=0;
  int fIsConnected = 0;

  fIoCtlSuccess = DeviceIoControl(
                    (HANDLE) OsHandle,          /* Handle to device  */
                    (ULONG) IOCTL_HIF_ISCONNECTED,  /* IO Control code for F-HIF card */
                    NULL,                       /* Buffer to driver. */
                    0,                          /* Length of buffer in bytes. */
                    &fIsConnected,              /* Buffer from driver.  */
                    sizeof(int),                /* Length of buffer in bytes. */
                    &dwReturned,                /* Bytes placed in DataBuffer. */
                    NULL                        /* NULL means wait till op. completes. */
                    );

  DBG_ApiFailureIsZero(fIoCtlSuccess);

  if (!fIoCtlSuccess || !fIsConnected)
  {
    return BX_E_NOT_CONNECTED;
  }

  return BX_E_OK;
}



bx_errtype BestXHIFIsDisconnected (bx_portnumtype OsHandle)
{
  BOOL fIoCtlSuccess;
  DWORD   dwReturned=0;
  int fIsDisconnected = 0;

  fIoCtlSuccess = DeviceIoControl(
                    (HANDLE) OsHandle,          /* Handle to device  */
                    (ULONG) IOCTL_HIF_ISDISCONNECTED,  /* IO Control code for F-HIF card */
                    NULL,                       /* Buffer to driver. */
                    0,                          /* Length of buffer in bytes. */
                    &fIsDisconnected,              /* Buffer from driver.  */
                    sizeof(int),                /* Length of buffer in bytes. */
                    &dwReturned,                /* Bytes placed in DataBuffer. */
                    NULL                        /* NULL means wait till op. completes. */
                    );

  DBG_ApiFailureIsZero(fIoCtlSuccess);

  /* NOTE:
     This IO control is typically called from WaitForClose which will wait
     until this function returns BX_E_NOT_CONNECTED. In the error case this
     will result in the BX_E_OK which gives the card enough time to finish
     this cycle */
  
  if (fIoCtlSuccess && fIsDisconnected)
  {
    return BX_E_NOT_CONNECTED;
  }

  return BX_E_OK;
}


/* --------------------------------------------------------------------------
 * Should only be called from BestXPortTimeoutSet()
 * -------------------------------------------------------------------------- */
bx_errtype BestXHifPortTimeoutSet(bx_portnumtype OsHandle, 
                                BESTTIMEOUTS * pCallersTimeouts)
{
  BOOL fIoCtlSuccess;
  ULONG ReturnedLength = 0;

  fIoCtlSuccess = DeviceIoControl(
    (HANDLE) OsHandle,          /* Handle to device  */
    (ULONG) IOCTL_HIF_SET_TIMEOUT, /* IO Control code for setting timeouts */
    (PVOID)pCallersTimeouts,    /* Buffer to driver. */
    sizeof(BESTTIMEOUTS),       /* Length of buffer in bytes. */
    NULL,                       /* Buffer from driver.  */
    0,                          /* Length of buffer in bytes. */
    &ReturnedLength,            /* Bytes placed in DataBuffer. */
    NULL                        /* NULL means wait till op. completes. */
    );

  DBG_ApiFailureIsZero(fIoCtlSuccess);

  if (!fIoCtlSuccess) 
  {
    return BX_E_API_ERROR;
  }
  
  return BX_E_OK;
}


/* --------------------------------------------------------------------------
 * Call with IOCTL_HIF_WRITECTRL (Win95 ONLY !!!)
 * or IOCTL_HIF_WRITE_DIRECT (Block mode; NT or Win95)
 * -------------------------------------------------------------------------- */

bx_errtype BestXHIFIoCtlWrite( bx_handletype handle, 
                             bx_int8 * pData,
                             bx_int32 NumBytes,
                             bx_int32 * pNumBytesWritten,
                             bx_int32 IoCtlCode)
{
  BOOL fIoCtlSuccess;
  HANDLE    OsHandle = bx_handlearray [handle].portnumber;

  assert(pData && pNumBytesWritten && "Null passed to BestXHIFIoCtlWrite()");

  /* init for failure */
  *pNumBytesWritten = 0;

  fIoCtlSuccess = DeviceIoControl(
    (HANDLE) OsHandle,          /* Handle to device  */
    (ULONG) IoCtlCode,          /* IO Control code for write */
    pData,                      /* Buffer to driver. */
    NumBytes,                   /* Length of buffer in bytes. */
    NULL,                       /* Buffer from driver.  */
    0,                          /* Length of buffer in bytes. */
    pNumBytesWritten,            /* Bytes placed in DataBuffer. */
    NULL                        /* NULL means wait till op. completes. */
    );

  DBG_ApiFailureIsZero(fIoCtlSuccess);
  BestXLastErrorParamSet (handle, BX_ERRPAR_1, BX_EERR_DRV_FAILURE);
  BestXLastErrorParamStringSet(handle,"A driver returned no success.");
  return fIoCtlSuccess ? BX_E_OK : BX_E_TRANSF_CMD;

}


/* --------------------------------------------------------------------------
 * Call with IOCTL_HIF_READCTRL (Win95 ONLY !!!)
 * or IOCTL_HIF_READ_DIRECT (Block mode; NT or Win95)
 * -------------------------------------------------------------------------- */

bx_errtype BestXHIFIoCtlRead(bx_handletype handle, 
                           bx_int8 * pData,
                           bx_int32 NumBytes,
                           bx_int32 * pNumBytesRead,
                           bx_int32 IoCtlCode)
{
  BOOL fIoCtlSuccess;
  HANDLE    OsHandle = bx_handlearray [handle].portnumber;

  assert(pData && pNumBytesRead && "Null passed to BestXHIFDirectRead()");

  fIoCtlSuccess = DeviceIoControl(
    (HANDLE) OsHandle,          /* Handle to device  */
    (ULONG) IoCtlCode,          /* IO Control code for Read */
    NULL,                       /* Buffer to driver. */
    0,                          /* Length of buffer in bytes. */
    pData,                      /* Buffer from driver.  */
    NumBytes,                   /* Length of buffer in bytes. */
    pNumBytesRead,              /* Bytes placed in DataBuffer. */
    NULL                        /* NULL means wait till op. completes. */
    );

  DBG_ApiFailureIsZero(fIoCtlSuccess);
  BestXLastErrorParamSet (handle, BX_ERRPAR_1, BX_EERR_DRV_FAILURE); 
  BestXLastErrorParamStringSet(handle,"A driver returned no success.");
  return fIoCtlSuccess ? BX_E_OK : BX_E_TRANSF_CMD;
}
